<?php
/**
 * Plugin Name: SMS Vert Pro - Envoi de SMS
 * Plugin URI: https://www.smsvertpro.com/
 * Description: Intégration de l'API SMS Vert Pro pour l'envoi de SMS depuis WordPress.
 * Version: 1.0.1
 * Author: 3 Bees Online
 * Author URI: https://www.3beesonline.com/
 * License: GPLv2 or later
 * Text Domain: smsvertpro-api
 * Requires at least: 5.0
 * Requires PHP: 7.4
 * Tested up to: 6.8
 */

// Sécurité de base
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// URL de l'API en dur
define('WP_SMSVERTPRO_API_URL', 'https://www.smsvertpro.com/api/');

// Version du plugin
define('WP_SMSVERTPRO_VERSION', '1.0.1');

// URL du serveur de mise à jour
define('WP_SMSVERTPRO_UPDATE_URL', 'https://www.smsvertpro.com/docs/update-info.json');

// Clé de chiffrement pour les données sensibles
// IMPORTANT : Ajouter cette constante dans wp-config.php pour plus de sécurité
if (!defined('WP_SMSVERTPRO_ENCRYPTION_KEY')) {
    define('WP_SMSVERTPRO_ENCRYPTION_KEY', 'smsvert_' . AUTH_KEY);
}

// Fonction pour chiffrer les données sensibles
function wp_smsvertpro_encrypt($data) {
    $cipher = 'AES-256-CBC';
    $key = hash('sha256', WP_SMSVERTPRO_ENCRYPTION_KEY);
    $iv = openssl_random_pseudo_bytes(openssl_cipher_iv_length($cipher));
    $encrypted = openssl_encrypt($data, $cipher, $key, 0, $iv);
    return base64_encode($encrypted . '::' . $iv);
}

// Fonction pour déchiffrer les données sensibles
function wp_smsvertpro_decrypt($data) {
    if (empty($data)) {
        return '';
    }
    $cipher = 'AES-256-CBC';
    $key = hash('sha256', WP_SMSVERTPRO_ENCRYPTION_KEY);
    list($encrypted_data, $iv) = array_pad(explode('::', base64_decode($data), 2), 2, null);
    if ($iv === null) {
        return '';
    }
    return openssl_decrypt($encrypted_data, $cipher, $key, 0, $iv);
}

// 1. Menu et sous-menus
add_action('admin_menu', function() {
    // Menu principal
    add_menu_page(
        'SMS Vert Pro',
        'SMS Vert Pro',
        'manage_options',
        'smsvertpro',
        'wp_smsvertpro_single_sms_page',
        'dashicons-format-chat',
        80
    );

    // Sous-menu : Envoyer un SMS
    add_submenu_page(
        'smsvertpro',
        'Envoyer un SMS',
        'Envoyer un SMS',
        'manage_options',
        'smsvertpro',
        'wp_smsvertpro_single_sms_page'
    );

    // Sous-menu : Envoi SMS en masse
    add_submenu_page(
        'smsvertpro',
        'Envoi SMS en masse',
        'Envoi SMS en masse',
        'manage_options',
        'smsvertpro-bulk',
        'wp_smsvertpro_bulk_sms_page'
    );

    // Sous-menu : Historique des campagnes
    add_submenu_page(
        'smsvertpro',
        'Historique des campagnes',
        'Historique',
        'manage_options',
        'smsvertpro-history',
        'wp_smsvertpro_history_page'
    );

    // Sous-menu : Créditer mon compte (lien externe)
    add_submenu_page(
        'smsvertpro',
        'Créditer mon compte',
        'Créditer mon compte',
        'manage_options',
        'smsvertpro-credits',
        'wp_smsvertpro_redirect_to_credits'
    );
});

// Fonction de redirection vers la page de crédit
function wp_smsvertpro_redirect_to_credits() {
    // Cette page ne sera jamais affichée car on intercepte le clic avec JavaScript
    echo '<p>Redirection vers la page de commande...</p>';
}

// Ajouter du JavaScript pour intercepter le clic sur le menu "Créditer mon compte"
add_action('admin_footer', function() {
    ?>
    <script type="text/javascript">
    jQuery(document).ready(function($) {
        // Intercepter le clic sur le lien "Créditer mon compte"
        $('a[href*="smsvertpro-credits"]').on('click', function(e) {
            e.preventDefault();
            window.open('https://www.smsvertpro.com/commander/', '_blank');
            return false;
        });
    });
    </script>
    <?php
});

// 2. Page : Envoyer un SMS
function wp_smsvertpro_single_sms_page() {
    if (!current_user_can('manage_options')) {
        return;
    }

    // Sauvegarde des options de connexion
    $settings_saved = false;
    $login_attempted = false;
    if (isset($_POST['wp_smsvertpro_save_settings']) && check_admin_referer('wp_smsvertpro_save_settings')) {
        $login_attempted = true;
        if (isset($_POST['wp_smsvertpro_email']) && !empty($_POST['wp_smsvertpro_email'])) {
            update_option('wp_smsvertpro_email', sanitize_email(wp_unslash($_POST['wp_smsvertpro_email'])));
        }
        if (isset($_POST['wp_smsvertpro_password']) && !empty($_POST['wp_smsvertpro_password'])) {
            $password_plain = sanitize_text_field(wp_unslash($_POST['wp_smsvertpro_password']));
            // Chiffrer le mot de passe avant de le stocker
            $password_encrypted = wp_smsvertpro_encrypt($password_plain);
            update_option('wp_smsvertpro_password', $password_encrypted);
        }

        if (isset($_POST['wp_smsvertpro_sender']) && !empty($_POST['wp_smsvertpro_sender'])) {
            update_option('wp_smsvertpro_sender', sanitize_text_field(wp_unslash($_POST['wp_smsvertpro_sender'])));
            $settings_saved = true;
        }
    }

    // Déconnexion
    if (isset($_POST['wp_smsvertpro_disconnect']) && check_admin_referer('wp_smsvertpro_disconnect')) {
        delete_option('wp_smsvertpro_email');
        delete_option('wp_smsvertpro_password');
        delete_option('wp_smsvertpro_sender');
        echo '<div class="notice notice-success is-dismissible"><p>Déconnecté avec succès.</p></div>';
    }

    $email    = get_option('wp_smsvertpro_email', '');
    $password = get_option('wp_smsvertpro_password', '');
    $sender   = get_option('wp_smsvertpro_sender', '');

    // Traiter l'envoi de SMS
    $sms_sent = false;
    $sms_error = false;
    if (isset($_POST['wp_smsvertpro_test_send']) && check_admin_referer('wp_smsvertpro_test_send')) {
        if (isset($_POST['wp_smsvertpro_sender'])) {
            $new_sender = sanitize_text_field(wp_unslash($_POST['wp_smsvertpro_sender']));
            update_option('wp_smsvertpro_sender', $new_sender);
            $sender = $new_sender;
        }

        $country_code = isset($_POST['wp_smsvertpro_country_code']) ? sanitize_text_field(wp_unslash($_POST['wp_smsvertpro_country_code'])) : '33';
        $phone_local  = isset($_POST['wp_smsvertpro_test_phone']) ? sanitize_text_field(wp_unslash($_POST['wp_smsvertpro_test_phone'])) : '';
        $phone_local = ltrim($phone_local, '0');
        $phone = $country_code . $phone_local;
        $message = isset($_POST['wp_smsvertpro_test_message']) ? sanitize_textarea_field(wp_unslash($_POST['wp_smsvertpro_test_message'])) : '';

        if (isset($_POST['wp_smsvertpro_add_stop'])) {
            if (stripos($message, 'STOP 36173') === false) {
                $message .= ' STOP 36173';
            }
        }

        if (wp_smsvertpro_envoyer_sms($phone, $message)) {
            $sms_sent = true;
        } else {
            $sms_error = true;
        }
    }

    // Vérifier la connexion et récupérer les crédits (après l'envoi éventuel)
    $is_connected = false;
    $credits = null;

    if (!empty($email) && !empty($password)) {
        $credits = wp_smsvertpro_get_credits();
        if ($credits !== false) {
            $is_connected = true;
        }
    }

    $logo_url = plugins_url('assets/smsvertpro-envoi-sms.png', __FILE__);

    ?>
    <style>
        /* Masquer les notices WordPress/Elementor sur cette page */
        .wrap .e-notice,
        .wrap .notice.is-dismissible:not(.notice-success):not(.notice-error) {
            display: none !important;
        }
        /* Augmenter la largeur max des cartes */
        .wrap .card {
            max-width: 100% !important;
        }
        /* Aligner les boutons submit à droite */
        .wrap p.submit {
            text-align: right !important;
        }
        /* Augmenter la hauteur de l'input tel */
        input[type="tel"] {
            height: 36px !important;
            line-height: 36px !important;
        }
    </style>
    <script>
        function toggleStopText() {
            const message = document.getElementById('wp_smsvertpro_test_message');
            const stopCheckbox = document.getElementById('wp_smsvertpro_add_stop');
            const stopText = ' STOP 36173';

            if (stopCheckbox.checked) {
                // Ajouter STOP si pas déjà présent
                if (!message.value.includes('STOP 36173')) {
                    message.value += stopText;
                }
            } else {
                // Retirer STOP si présent
                message.value = message.value.replace(stopText, '').replace('STOP 36173', '');
            }

            updateSmsCounter();
        }

        function updateSmsCounter() {
            const message = document.getElementById('wp_smsvertpro_test_message');
            const counter = document.getElementById('sms_counter');
            const length = message.value.length;

            let credits = 0;
            if (length === 0) {
                credits = 0;
            } else if (length <= 160) {
                credits = 1;
            } else {
                // Premier SMS = 160 caractères, puis tranches de 153
                credits = 1 + Math.ceil((length - 160) / 153);
            }

            counter.innerHTML = '<strong>' + length + '</strong> caractères = <strong>' + credits + '</strong> crédit' + (credits > 1 ? 's' : '') + ' SMS';
        }
    </script>
    <div class="wrap">
        <?php if (!$is_connected): ?>
            <!-- Logo centré -->
            <div style="text-align: center; margin: 40px 0;">
                <img src="<?php echo esc_url($logo_url); ?>" alt="SMS Vert Pro" style="max-height: 80px;">
            </div>

            <!-- Formulaire de connexion centré -->
            <div style="max-width: 600px; margin: 0 auto;">
                <div class="card">
                    <h2>Connexion à l'API SMS Vert Pro</h2>
                    <p>Veuillez vous connecter à votre compte SMS Vert Pro pour accéder aux fonctionnalités d'envoi de SMS.</p>

                    <?php if ($login_attempted && !empty($email) && !empty($password) && $credits === false): ?>
                        <div class="notice notice-error">
                            <p><strong>Erreur de connexion.</strong> Vérifiez vos identifiants.</p>
                        </div>
                    <?php endif; ?>

                    <form method="post">
                        <?php wp_nonce_field('wp_smsvertpro_save_settings'); ?>
                        <table class="form-table">
                            <tr>
                                <th scope="row"><label for="wp_smsvertpro_email">Email (Login)</label></th>
                                <td><input name="wp_smsvertpro_email" id="wp_smsvertpro_email" type="email" class="regular-text" value="<?php echo esc_attr($email); ?>" required></td>
                            </tr>
                            <tr>
                                <th scope="row"><label for="wp_smsvertpro_password">Mot de passe</label></th>
                                <td><input name="wp_smsvertpro_password" id="wp_smsvertpro_password" type="password" class="regular-text" value="" required></td>
                            </tr>
                        </table>
                        <?php submit_button('Se connecter', 'primary', 'wp_smsvertpro_save_settings'); ?>
                    </form>
                </div>

                <!-- Bloc inscription -->
                <div class="card" style="margin-top: 20px;">
                    <h3>Pas encore de compte SMS Vert Pro ?</h3>
                    <p>Créez votre compte pour commencer à envoyer des SMS depuis WordPress.</p>
                    <p>
                        <a href="https://www.smsvertpro.com/espace-client/?type=1&wordpress=1" class="button button-secondary" target="_blank">
                            Créer un compte SMS Vert Pro
                        </a>
                    </p>
                    <p class="description" style="margin-top: 15px;">
                        <strong>Note :</strong> Votre compte sera activé après validation de vos données par notre équipe.
                    </p>
                </div>
            </div>

        <?php else: ?>
            <!-- Logo centré -->
            <div style="text-align: center; margin: 20px 0;">
                <img src="<?php echo esc_url($logo_url); ?>" alt="SMS Vert Pro" style="max-height: 60px;">
            </div>

            <!-- Interface connectée centrée -->
            <div style="max-width: 1200px; margin: 0 auto; padding: 0 20px;">
                <div class="notice notice-success" style="display: flex; align-items: center; justify-content: space-between;">
                    <p>
                        <strong>✓ Connecté</strong> &nbsp;|&nbsp;
                        <strong>Crédits SMS disponibles : <?php echo esc_html($credits); ?></strong> &nbsp;|&nbsp;
                        Compte : <?php echo esc_html($email); ?>
                    </p>
                    <form method="post" style="margin: 0;">
                        <?php wp_nonce_field('wp_smsvertpro_disconnect'); ?>
                        <button type="submit" name="wp_smsvertpro_disconnect" class="button">Se déconnecter</button>
                    </form>
                </div>

                <?php if ($settings_saved): ?>
                    <div class="notice notice-success is-dismissible">
                        <p>Paramètres enregistrés avec succès.</p>
                    </div>
                <?php endif; ?>

                <!-- Envoi SMS -->
                <div style="max-width: 900px; margin: 0 auto;">
                    <h2>Envoyer un SMS</h2>

                    <?php if ($sms_sent): ?>
                        <div class="notice notice-success"><p><strong>✓ SMS envoyé avec succès !</strong></p></div>
                    <?php elseif ($sms_error): ?>
                        <div class="notice notice-error"><p><strong>✗ Erreur lors de l'envoi du SMS.</strong> Vérifiez le numéro et votre solde de crédits.</p></div>
                    <?php endif; ?>

                    <div class="card">
                    <form method="post">
                        <?php wp_nonce_field('wp_smsvertpro_test_send'); ?>
                        <table class="form-table">
                            <tr>
                                <th scope="row"><label for="wp_smsvertpro_sender">Expéditeur</label></th>
                                <td>
                                    <input name="wp_smsvertpro_sender" id="wp_smsvertpro_sender" type="text" class="regular-text" value="<?php echo esc_attr($sender); ?>" maxlength="11" required>
                                    <p class="description">Nom qui apparaîtra comme expéditeur (11 caractères max, pas uniquement des chiffres)</p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><label for="wp_smsvertpro_test_phone">Numéro de téléphone</label></th>
                                <td>
                                    <div style="display: flex; gap: 10px; align-items: flex-start;">
                                        <select name="wp_smsvertpro_country_code" id="wp_smsvertpro_country_code" style="width: 200px; height: 36px;">
                                            <option value="33" selected>🇫🇷 France (+33)</option>
                                            <option value="32" disabled>🇧🇪 Belgique (+32)</option>
                                            <option value="41" disabled>🇨🇭 Suisse (+41)</option>
                                            <option value="352" disabled>🇱🇺 Luxembourg (+352)</option>
                                        </select>
                                        <div style="flex: 1;">
                                            <input type="tel" name="wp_smsvertpro_test_phone" id="wp_smsvertpro_test_phone" class="regular-text" placeholder="06 12 34 56 78" pattern="0[67][0-9]{8}" required>
                                            <p class="description">Format : 06 ou 07 suivi de 8 chiffres</p>
                                        </div>
                                    </div>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><label for="wp_smsvertpro_test_message">Message</label></th>
                                <td>
                                    <textarea name="wp_smsvertpro_test_message" id="wp_smsvertpro_test_message" rows="4" class="large-text" placeholder="Votre message ici..." required oninput="updateSmsCounter()"></textarea>
                                    <p class="description" id="sms_counter"><strong>0</strong> caractères = <strong>0</strong> crédit SMS</p>
                                    <p class="description" style="margin-top: 5px; font-style: italic;">
                                        ℹ️ Un crédit SMS contient 160 caractères.<br />Les SMS supplémentaires sont débités par tranche de 153 caractères.
                                    </p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"></th>
                                <td>
                                    <label>
                                        <input type="checkbox" name="wp_smsvertpro_add_stop" id="wp_smsvertpro_add_stop" value="1" required onchange="toggleStopText()">
                                        <strong>Ajouter "STOP 36173" à la fin du message (obligatoire)</strong>
                                    </label>
                                    <p class="description">Cette mention de désabonnement est obligatoire pour l'envoi de SMS commerciaux.</p>
                                </td>
                            </tr>
                        </table>
                        <?php submit_button('Envoyer le SMS', 'primary', 'wp_smsvertpro_test_send'); ?>
                    </form>
                    </div><!-- Fin .card -->
                </div><!-- Fin conteneur 900px -->

            </div><!-- Fin du conteneur 1200px -->
        <?php endif; ?>
    </div>
    <?php
}

// 3. Page : Envoi SMS en masse
function wp_smsvertpro_bulk_sms_page() {
    if (!current_user_can('manage_options')) {
        return;
    }

    // Déconnexion
    if (isset($_POST['wp_smsvertpro_disconnect']) && check_admin_referer('wp_smsvertpro_disconnect')) {
        delete_option('wp_smsvertpro_email');
        delete_option('wp_smsvertpro_password');
        delete_option('wp_smsvertpro_sender');
        echo '<div class="notice notice-success is-dismissible"><p>Déconnecté avec succès.</p></div>';
    }

    $email    = get_option('wp_smsvertpro_email', '');
    $password = get_option('wp_smsvertpro_password', '');
    $sender   = get_option('wp_smsvertpro_sender', '');

    // Traiter l'envoi de SMS en masse
    $bulk_results = null;
    if (isset($_POST['wp_smsvertpro_bulk_send']) && check_admin_referer('wp_smsvertpro_bulk_send')) {
        if (isset($_POST['wp_smsvertpro_bulk_sender'])) {
            $bulk_sender = sanitize_text_field(wp_unslash($_POST['wp_smsvertpro_bulk_sender']));
            update_option('wp_smsvertpro_sender', $bulk_sender);
            $sender = $bulk_sender;
        }

        $recipient_type = isset($_POST['wp_smsvertpro_recipient_type']) ? sanitize_text_field(wp_unslash($_POST['wp_smsvertpro_recipient_type'])) : '';
        $message = isset($_POST['wp_smsvertpro_bulk_message']) ? sanitize_textarea_field(wp_unslash($_POST['wp_smsvertpro_bulk_message'])) : '';

        if (isset($_POST['wp_smsvertpro_bulk_add_stop'])) {
            if (stripos($message, 'STOP 36173') === false) {
                $message .= ' STOP 36173';
            }
        }

        $phone_numbers = [];
        switch ($recipient_type) {
            case 'all_users':
                $phone_numbers = wp_smsvertpro_get_all_users_phones(false);
                break;
            case 'woo_customers':
                $phone_numbers = wp_smsvertpro_get_woo_customers_phones(false, false);
                break;
            case 'cf7_contacts':
                $phone_numbers = wp_smsvertpro_get_cf7_contacts(false);
                break;
            case 'custom_list':
                $custom_numbers = isset($_POST['wp_smsvertpro_custom_numbers']) ? sanitize_textarea_field(wp_unslash($_POST['wp_smsvertpro_custom_numbers'])) : '';
                $phone_numbers = wp_smsvertpro_parse_custom_numbers($custom_numbers);
                break;
        }

        $bulk_results = wp_smsvertpro_send_bulk_sms($phone_numbers, $message);
    }

    // Vérifier la connexion et récupérer les crédits
    $is_connected = false;
    $credits = null;
    if (!empty($email) && !empty($password)) {
        $credits = wp_smsvertpro_get_credits();
        if ($credits !== false) {
            $is_connected = true;
        }
    }

    // Calculer le nombre de contacts pour chaque option
    $all_users_count = 0;
    $woo_customers_count = 0;
    $woo_customers_orders_count = 0;
    $cf7_contacts_count = 0;
    $all_users_phones = [];
    $woo_customers_phones = [];
    $cf7_contacts_phones = [];
    $all_users_with_names = [];
    $woo_customers_with_names = [];
    $cf7_contacts_with_names = [];

    if ($is_connected) {
        $all_users_phones = wp_smsvertpro_get_all_users_phones(false);
        $all_users_count = count($all_users_phones);
        $all_users_with_names = wp_smsvertpro_get_all_users_phones(true);

        if (class_exists('WooCommerce')) {
            $woo_customers_phones = wp_smsvertpro_get_woo_customers_phones(false, false);
            $woo_customers_count = count($woo_customers_phones);
            $woo_customers_orders_count = count(wp_smsvertpro_get_woo_customers_phones(true, false));
            $woo_customers_with_names = wp_smsvertpro_get_woo_customers_phones(false, true);
        }

        // Contacts depuis formulaires CF7
        $cf7_contacts_phones = wp_smsvertpro_get_cf7_contacts(false);
        $cf7_contacts_count = count($cf7_contacts_phones);
        $cf7_contacts_with_names = wp_smsvertpro_get_cf7_contacts(true);
    }

    $logo_url = plugins_url('assets/smsvertpro-envoi-sms.png', __FILE__);
    ?>
    <style>
        .wrap .e-notice,
        .wrap .notice.is-dismissible:not(.notice-success):not(.notice-error) {
            display: none !important;
        }
        .wrap .card {
            max-width: 100% !important;
        }
        .wrap p.submit {
            text-align: right !important;
        }
        input[type="tel"] {
            height: 36px !important;
            line-height: 36px !important;
        }
        /* Style pour bouton désactivé */
        input[type="submit"]:disabled,
        input[type="submit"].disabled {
            opacity: 0.5 !important;
            cursor: not-allowed !important;
            pointer-events: none !important;
        }
    </style>
    <div class="wrap">
        <?php if (!$is_connected): ?>
            <div style="text-align: center; margin: 40px 0;">
                <img src="<?php echo esc_url($logo_url); ?>" alt="SMS Vert Pro" style="max-height: 80px;">
            </div>
            <div style="max-width: 600px; margin: 0 auto;">
                <div class="card">
                    <h2>Connexion requise</h2>
                    <p>Vous devez d'abord vous connecter à votre compte SMS Vert Pro pour accéder à l'envoi de SMS en masse.</p>
                    <p><a href="<?php echo esc_url(admin_url('admin.php?page=smsvertpro')); ?>" class="button button-primary">Se connecter</a></p>
                </div>
            </div>
        <?php else: ?>
            <div style="text-align: center; margin: 20px 0;">
                <img src="<?php echo esc_url($logo_url); ?>" alt="SMS Vert Pro" style="max-height: 60px;">
            </div>

            <div style="max-width: 1200px; margin: 0 auto; padding: 0 20px;">
                <div class="notice notice-success" style="display: flex; align-items: center; justify-content: space-between;">
                    <p>
                        <strong>✓ Connecté</strong> &nbsp;|&nbsp;
                        <strong>Crédits SMS disponibles : <?php echo esc_html($credits); ?></strong> &nbsp;|&nbsp;
                        Compte : <?php echo esc_html($email); ?>
                    </p>
                    <form method="post" style="margin: 0;">
                        <?php wp_nonce_field('wp_smsvertpro_disconnect'); ?>
                        <button type="submit" name="wp_smsvertpro_disconnect" class="button">Se déconnecter</button>
                    </form>
                </div>

                <!-- Envoi SMS en masse -->
                <div style="max-width: 900px; margin: 0 auto;">
                    <h2>Envoi SMS en masse</h2>

                    <?php if ($bulk_results !== null): ?>
                        <?php if ($bulk_results['success'] > 0): ?>
                            <div class="notice notice-success">
                                <p><strong>✓ Envoi en masse terminé !</strong></p>
                                <p>
                                    <strong><?php echo absint($bulk_results['success']); ?></strong> SMS envoyé(s) avec succès sur <strong><?php echo absint($bulk_results['total']); ?></strong> destinataire(s).
                                    <?php if ($bulk_results['failed'] > 0): ?>
                                        <br><strong><?php echo absint($bulk_results['failed']); ?></strong> échec(s).
                                    <?php endif; ?>
                                </p>
                            </div>
                        <?php else: ?>
                            <div class="notice notice-error">
                                <p><strong>✗ Échec de l'envoi en masse</strong></p>
                                <p>Aucun SMS n'a pu être envoyé.</p>
                            </div>
                        <?php endif; ?>

                        <?php if (!empty($bulk_results['errors']) && count($bulk_results['errors']) <= 10): ?>
                            <div class="notice notice-warning">
                                <p><strong>Détails des erreurs :</strong></p>
                                <ul style="margin-left: 20px;">
                                    <?php foreach ($bulk_results['errors'] as $error): ?>
                                        <li><?php echo esc_html($error); ?></li>
                                    <?php endforeach; ?>
                                </ul>
                            </div>
                        <?php elseif (!empty($bulk_results['errors']) && count($bulk_results['errors']) > 10): ?>
                            <div class="notice notice-warning">
                                <p><strong><?php echo count($bulk_results['errors']); ?> erreur(s) détectée(s).</strong> (Liste tronquée pour plus de lisibilité)</p>
                            </div>
                        <?php endif; ?>
                    <?php endif; ?>

                    <div class="card">
                        <form method="post">
                            <?php wp_nonce_field('wp_smsvertpro_bulk_send'); ?>
                            <table class="form-table">
                                <tr>
                                    <th scope="row"><label for="wp_smsvertpro_recipient_type">Destinataires</label></th>
                                    <td>
                                        <select name="wp_smsvertpro_recipient_type" id="wp_smsvertpro_recipient_type" class="regular-text" required onchange="updateBulkPreview()">
                                            <option value="">-- Sélectionner --</option>
                                            <?php if (class_exists('WooCommerce')): ?>
                                                <option value="woo_customers">Clients WooCommerce (<?php echo absint($woo_customers_count); ?> contact<?php echo $woo_customers_count > 1 ? 's' : ''; ?>)</option>
                                            <?php endif; ?>
                                            <option value="cf7_contacts">Contacts formulaires (<?php echo absint($cf7_contacts_count); ?> contact<?php echo $cf7_contacts_count > 1 ? 's' : ''; ?>)</option>
                                            <option value="all_users">Utilisateurs WordPress (<?php echo absint($all_users_count); ?> contact<?php echo $all_users_count > 1 ? 's' : ''; ?>)</option>
                                            <option value="custom_list">Liste personnalisée</option>
                                        </select>
                                        <p class="description">Sélectionnez le groupe de destinataires pour l'envoi</p>
                                        <div id="phone_numbers_preview" style="display: none; margin-top: 15px; padding: 10px; background: #f9f9f9; border-left: 4px solid #2271b1;">
                                            <p style="margin: 0 0 10px 0;"><strong>📱 Aperçu des numéros :</strong></p>
                                            <div id="phone_numbers_list" style="max-height: 200px; overflow-y: auto; font-family: monospace; font-size: 12px;"></div>
                                        </div>
                                    </td>
                                </tr>
                                <tr id="custom_list_row" style="display: none;">
                                    <th scope="row"><label for="wp_smsvertpro_custom_numbers">Numéros (un par ligne)</label></th>
                                    <td>
                                        <textarea name="wp_smsvertpro_custom_numbers" id="wp_smsvertpro_custom_numbers" rows="6" class="large-text" placeholder="0612345678&#10;0687654321&#10;..." oninput="updateBulkPreview()"></textarea>
                                        <p class="description">Entrez les numéros au format 06 ou 07 (un par ligne)</p>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><label for="wp_smsvertpro_bulk_sender">Expéditeur</label></th>
                                    <td>
                                        <input name="wp_smsvertpro_bulk_sender" id="wp_smsvertpro_bulk_sender" type="text" class="regular-text" value="<?php echo esc_attr($sender); ?>" maxlength="11" required>
                                        <p class="description">Nom qui apparaîtra comme expéditeur</p>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><label for="wp_smsvertpro_bulk_message">Message</label></th>
                                    <td>
                                        <textarea name="wp_smsvertpro_bulk_message" id="wp_smsvertpro_bulk_message" rows="4" class="large-text" placeholder="Votre message ici..." required oninput="updateBulkSmsCounter()"></textarea>
                                        <p class="description" id="bulk_sms_counter"><strong>0</strong> caractères = <strong>0</strong> crédit SMS</p>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"></th>
                                    <td>
                                        <label>
                                            <input type="checkbox" name="wp_smsvertpro_bulk_add_stop" id="wp_smsvertpro_bulk_add_stop" value="1" required onchange="toggleBulkStopText()">
                                            <strong>Ajouter "STOP 36173" à la fin du message (obligatoire)</strong>
                                        </label>
                                        <p class="description">Cette mention de désabonnement est obligatoire pour l'envoi de SMS commerciaux.</p>
                                    </td>
                                </tr>
                            </table>
                            <div id="bulk_preview" style="display: none; background: #f0f0f1; padding: 15px; margin: 20px 0; border-left: 4px solid #2271b1;">
                                <p><strong>Aperçu de l'envoi :</strong></p>
                                <p id="bulk_preview_text"></p>
                            </div>
                            <div id="bulk_error" style="display: none; background: #f8d7da; color: #721c24; padding: 15px; margin: 20px 0; border-left: 4px solid #f5c6cb;">
                                <p style="margin: 0;"><strong>⚠️ Erreur :</strong> <span id="bulk_error_text"></span></p>
                            </div>
                            <?php submit_button('Envoyer aux destinataires', 'primary', 'wp_smsvertpro_bulk_send'); ?>
                        </form>
                    </div>
                </div>

            </div><!-- Fin du conteneur 1200px -->

            <script>
                // Données des numéros de téléphone depuis PHP (avec noms)
                const phoneNumbersData = {
                    'all_users': <?php echo json_encode($all_users_with_names); ?>,
                    'woo_customers': <?php echo json_encode($woo_customers_with_names); ?>,
                    'cf7_contacts': <?php echo json_encode($cf7_contacts_with_names); ?>
                };

                // Compteurs de contacts
                const contactCounts = {
                    'all_users': <?php echo absint($all_users_count); ?>,
                    'woo_customers': <?php echo absint($woo_customers_count); ?>,
                    'cf7_contacts': <?php echo absint($cf7_contacts_count); ?>
                };

                // Gestion de l'affichage de la liste personnalisée
                document.getElementById('wp_smsvertpro_recipient_type').addEventListener('change', function() {
                    const customRow = document.getElementById('custom_list_row');
                    if (this.value === 'custom_list') {
                        customRow.style.display = 'table-row';
                    } else {
                        customRow.style.display = 'none';
                    }
                    updateBulkPreview();
                    updatePhoneNumbersPreview();
                });

                function toggleBulkStopText() {
                    const message = document.getElementById('wp_smsvertpro_bulk_message');
                    const stopCheckbox = document.getElementById('wp_smsvertpro_bulk_add_stop');
                    const stopText = ' STOP 36173';

                    if (stopCheckbox.checked) {
                        if (!message.value.includes('STOP 36173')) {
                            message.value += stopText;
                        }
                    } else {
                        message.value = message.value.replace(stopText, '').replace('STOP 36173', '');
                    }

                    updateBulkSmsCounter();
                }

                function updateBulkSmsCounter() {
                    const message = document.getElementById('wp_smsvertpro_bulk_message');
                    const counter = document.getElementById('bulk_sms_counter');
                    const length = message.value.length;

                    let credits = 0;
                    if (length === 0) {
                        credits = 0;
                    } else if (length <= 160) {
                        credits = 1;
                    } else {
                        credits = 1 + Math.ceil((length - 160) / 153);
                    }

                    counter.innerHTML = '<strong>' + length + '</strong> caractères = <strong>' + credits + '</strong> crédit' + (credits > 1 ? 's' : '') + ' SMS';
                    updateBulkPreview();
                }

                function updateBulkPreview() {
                    const recipientType = document.getElementById('wp_smsvertpro_recipient_type').value;
                    const message = document.getElementById('wp_smsvertpro_bulk_message').value;
                    const preview = document.getElementById('bulk_preview');
                    const previewText = document.getElementById('bulk_preview_text');
                    const errorBox = document.getElementById('bulk_error');
                    const errorText = document.getElementById('bulk_error_text');
                    const submitButton = document.querySelector('input[name="wp_smsvertpro_bulk_send"]');

                    // Vérifier si la liste sélectionnée contient 0 contacts
                    if (recipientType && recipientType !== 'custom_list') {
                        const count = contactCounts[recipientType] || 0;
                        if (count === 0) {
                            errorBox.style.display = 'block';
                            errorText.innerHTML = 'La liste sélectionnée ne contient aucun contact.';
                            preview.style.display = 'none';
                            if (submitButton) {
                                submitButton.disabled = true;
                                submitButton.classList.add('disabled');
                            }
                            return;
                        }
                    }

                    errorBox.style.display = 'none';
                    if (submitButton) {
                        submitButton.disabled = false;
                        submitButton.classList.remove('disabled');
                    }

                    if (recipientType && message) {
                        preview.style.display = 'block';
                        let recipientLabel = '';
                        switch(recipientType) {
                            case 'all_users':
                                recipientLabel = 'tous les utilisateurs WordPress';
                                break;
                            case 'woo_customers':
                                recipientLabel = 'tous les clients WooCommerce';
                                break;
                            case 'cf7_contacts':
                                recipientLabel = 'tous les contacts formulaires';
                                break;
                            case 'custom_list':
                                const customNumbers = document.getElementById('wp_smsvertpro_custom_numbers').value;
                                const count = customNumbers.split('\n').filter(n => n.trim()).length;
                                recipientLabel = count + ' numéro(s) personnalisé(s)';
                                break;
                        }
                        previewText.innerHTML = 'Le message sera envoyé à <strong>' + recipientLabel + '</strong>.';
                    } else {
                        preview.style.display = 'none';
                    }
                }

                function updatePhoneNumbersPreview() {
                    const recipientType = document.getElementById('wp_smsvertpro_recipient_type').value;
                    const previewContainer = document.getElementById('phone_numbers_preview');
                    const phoneNumbersList = document.getElementById('phone_numbers_list');

                    if (!recipientType || recipientType === 'custom_list') {
                        previewContainer.style.display = 'none';
                        return;
                    }

                    let contacts = [];
                    if (phoneNumbersData[recipientType]) {
                        contacts = phoneNumbersData[recipientType];
                    }

                    if (contacts.length > 0) {
                        previewContainer.style.display = 'block';
                        let html = '<ul style="margin: 0; padding-left: 20px; list-style: none;">';
                        contacts.forEach(function(contact, index) {
                            // Formater le numéro : 33612345678 -> +33 6 12 34 56 78
                            let phone = contact.phone;
                            let formatted = '+' + phone.substring(0, 2) + ' ' +
                                           phone.substring(2, 3) + ' ' +
                                           phone.substring(3, 5) + ' ' +
                                           phone.substring(5, 7) + ' ' +
                                           phone.substring(7, 9) + ' ' +
                                           phone.substring(9, 11);
                            html += '<li style="padding: 3px 0; display: flex; justify-content: space-between;">';
                            html += '<span><strong>' + contact.name + '</strong></span>';
                            html += '<span style="color: #666;">' + formatted + '</span>';
                            html += '</li>';
                        });
                        html += '</ul>';
                        phoneNumbersList.innerHTML = html;
                    } else {
                        previewContainer.style.display = 'none';
                    }
                }

                // Mettre à jour l'aperçu pour la liste personnalisée
                document.getElementById('wp_smsvertpro_custom_numbers').addEventListener('input', function() {
                    updateBulkPreview();
                });

                // Validation avant soumission du formulaire
                document.querySelector('form').addEventListener('submit', function(e) {
                    const recipientType = document.getElementById('wp_smsvertpro_recipient_type').value;
                    const errorBox = document.getElementById('bulk_error');
                    const errorText = document.getElementById('bulk_error_text');

                    // Vérifier si 0 contacts pour les listes prédéfinies
                    if (recipientType && recipientType !== 'custom_list') {
                        const count = contactCounts[recipientType] || 0;
                        if (count === 0) {
                            e.preventDefault();
                            errorBox.style.display = 'block';
                            errorText.innerHTML = 'Impossible d\'envoyer : la liste sélectionnée ne contient aucun contact.';
                            errorBox.scrollIntoView({ behavior: 'smooth', block: 'center' });
                            return false;
                        }
                    }

                    // Vérifier si liste personnalisée est vide
                    if (recipientType === 'custom_list') {
                        const customNumbers = document.getElementById('wp_smsvertpro_custom_numbers').value.trim();
                        if (customNumbers === '') {
                            e.preventDefault();
                            errorBox.style.display = 'block';
                            errorText.innerHTML = 'Veuillez entrer au moins un numéro de téléphone dans la liste personnalisée.';
                            errorBox.scrollIntoView({ behavior: 'smooth', block: 'center' });
                            return false;
                        }
                    }
                });
            </script>
        <?php endif; ?>
    </div>
    <?php
}

// 2. Fonction pour récupérer les crédits
function wp_smsvertpro_get_credits() {
    $email    = get_option('wp_smsvertpro_email');
    $password_encrypted = get_option('wp_smsvertpro_password');

    if (empty($email) || empty($password_encrypted)) {
        return false;
    }

    // Déchiffrer le mot de passe
    $password = wp_smsvertpro_decrypt($password_encrypted);
    if (empty($password)) {
        return false;
    }

    // Créer le flux XML avec l'identifiant du plugin
    $plugin_id = 'Plugin WP ' . WP_SMSVERTPRO_VERSION;
    $xml = '<?xml version="1.0" encoding="UTF-8"?>';
    $xml .= '<sms>';
    $xml .= '<login>';
    $xml .= '<user>' . esc_html($email) . '</user>';
    $xml .= '<pass>' . md5($password) . '</pass>';
    $xml .= '<id>' . esc_html($plugin_id) . '</id>';
    $xml .= '</login>';
    $xml .= '</sms>';

    // Envoyer la requête POST
    $response = wp_remote_post(WP_SMSVERTPRO_API_URL . 'credits', [
        'timeout' => 15,
        'headers' => [
            'Content-Type' => 'application/xml',
        ],
        'body' => $xml,
    ]);

    if (is_wp_error($response)) {
        return false;
    }

    $code = wp_remote_retrieve_response_code($response);
    if ($code !== 200) {
        return false;
    }

    // Parser la réponse XML
    $body = wp_remote_retrieve_body($response);

    // Supprimer les caractères de contrôle invisibles
    $body = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/', '', $body);

    // Charger le XML
    libxml_use_internal_errors(true);
    $xml_response = simplexml_load_string($body);

    if ($xml_response === false) {
        return false;
    }

    // Récupérer le nombre de crédits
    if (isset($xml_response->credits)) {
        return (int) $xml_response->credits;
    }

    return false;
}

// 3. Fonction d'envoi via l'API
function wp_smsvertpro_envoyer_sms($telephone, $message) {
    $email    = get_option('wp_smsvertpro_email');
    $password_encrypted = get_option('wp_smsvertpro_password');
    $sender   = get_option('wp_smsvertpro_sender');

    if (empty($email) || empty($password_encrypted)) {
        return false;
    }

    // Déchiffrer le mot de passe
    $password = wp_smsvertpro_decrypt($password_encrypted);
    if (empty($password)) {
        return false;
    }

    // Créer le flux XML
    $xml = '<?xml version="1.0" encoding="UTF-8"?>';
    $xml .= '<sms>';
    $xml .= '<login>';
    $xml .= '<user>' . esc_html($email) . '</user>';
    $xml .= '<pass>' . md5($password) . '</pass>';
    $xml .= '</login>';
    $xml .= '<message id="">';
    $xml .= '<sender>' . esc_html($sender) . '</sender>';
    $xml .= '<text>' . esc_html($message) . '</text>';
    $xml .= '<delay cancel=""></delay>';
    $xml .= '</message>';
    $xml .= '<recipients>';
    $xml .= '<gsm id="">' . esc_html($telephone) . '</gsm>';
    $xml .= '</recipients>';
    $xml .= '</sms>';

    $response = wp_remote_post(WP_SMSVERTPRO_API_URL . 'sending', [
        'timeout' => 15,
        'headers' => [
            'Content-Type' => 'application/xml',
        ],
        'body' => $xml,
    ]);

    if (is_wp_error($response)) {
        return false;
    }

    $code = wp_remote_retrieve_response_code($response);
    if ($code !== 200) {
        return false;
    }

    // Parser la réponse XML
    $body = wp_remote_retrieve_body($response);
    libxml_use_internal_errors(true);
    $xml_response = simplexml_load_string($body);

    if ($xml_response === false) {
        return false;
    }

    // Vérifier le statut (SEND_OK)
    if (isset($xml_response->status) && (string)$xml_response->status === 'SEND_OK') {
        return true;
    }

    return false;
}

// 4. Fonction pour récupérer les numéros de tous les utilisateurs WordPress (SANS WooCommerce)
function wp_smsvertpro_get_all_users_phones($with_names = false) {
    $phones = [];
    $users = get_users(['fields' => ['ID', 'display_name', 'first_name', 'last_name']]);

    foreach ($users as $user) {
        // Chercher dans les meta : phone, mobile (mais PAS billing_phone qui est WooCommerce)
        $phone = get_user_meta($user->ID, 'phone', true);
        if (empty($phone)) {
            $phone = get_user_meta($user->ID, 'mobile', true);
        }

        if (!empty($phone)) {
            $formatted = wp_smsvertpro_format_phone($phone);
            if (!empty($formatted)) {
                if ($with_names) {
                    $name = trim($user->first_name . ' ' . $user->last_name);
                    if (empty($name)) {
                        $name = $user->display_name;
                    }
                    $phones[] = ['phone' => $formatted, 'name' => $name];
                } else {
                    $phones[] = $formatted;
                }
            }
        }
    }

    if ($with_names) {
        // Trier par ordre alphabétique sur le nom
        usort($phones, function($a, $b) {
            return strcasecmp($a['name'], $b['name']);
        });
        return $phones;
    }

    return array_filter(array_unique($phones)); // Retirer les doublons et valeurs vides
}

// 5. Fonction pour récupérer les numéros des clients WooCommerce
function wp_smsvertpro_get_woo_customers_phones($with_orders_only = false, $with_names = false) {
    $phones = [];

    if (!class_exists('WooCommerce')) {
        return $phones;
    }

    global $wpdb;

    // Détecter si HPOS (High-Performance Order Storage) est activé
    $hpos_enabled = false;
    if (class_exists('Automattic\WooCommerce\Utilities\OrderUtil')) {
        $hpos_enabled = \Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled();
    }

    $billing_data = [];

    if ($hpos_enabled) {
        // HPOS activé : utiliser les tables personnalisées
        // Récupérer phone, first_name et last_name depuis wc_order_addresses
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Complex query for WooCommerce HPOS, no equivalent function
        $billing_data = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT DISTINCT addr.phone, addr.first_name, addr.last_name
                FROM {$wpdb->prefix}wc_order_addresses addr
                INNER JOIN {$wpdb->prefix}wc_orders o ON o.id = addr.order_id
                WHERE addr.address_type = %s
                AND addr.phone != ''
                AND addr.phone IS NOT NULL
                AND o.status NOT IN ('trash', 'auto-draft')",
                'billing'
            ),
            ARRAY_A
        );
    } else {
        // Mode traditionnel : récupérer depuis postmeta
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Complex query for WooCommerce orders, no equivalent function
        $order_ids = $wpdb->get_col(
            $wpdb->prepare(
                "SELECT DISTINCT p.ID
                FROM {$wpdb->prefix}posts p
                INNER JOIN {$wpdb->prefix}postmeta pm ON p.ID = pm.post_id
                WHERE p.post_type = %s
                AND p.post_status NOT IN ('trash', 'auto-draft')
                AND pm.meta_key = %s
                AND pm.meta_value != ''",
                'shop_order',
                '_billing_phone'
            )
        );

        foreach ($order_ids as $order_id) {
            $phone = get_post_meta($order_id, '_billing_phone', true);
            $first_name = get_post_meta($order_id, '_billing_first_name', true);
            $last_name = get_post_meta($order_id, '_billing_last_name', true);

            if (!empty($phone)) {
                $billing_data[] = [
                    'phone' => $phone,
                    'first_name' => $first_name,
                    'last_name' => $last_name
                ];
            }
        }
    }

    // Traiter les données et éviter les doublons
    $seen_phones = [];
    foreach ($billing_data as $data) {
        $phone = $data['phone'];
        $formatted = wp_smsvertpro_format_phone($phone);

        if (!empty($formatted) && !in_array($formatted, $seen_phones)) {
            $seen_phones[] = $formatted;

            if ($with_names) {
                $name = trim($data['first_name'] . ' ' . $data['last_name']);
                if (empty($name)) {
                    $name = 'Client';
                }
                $phones[] = ['phone' => $formatted, 'name' => $name];
            } else {
                $phones[] = $formatted;
            }
        }
    }

    // Trier par ordre alphabétique sur le nom si demandé
    if ($with_names && !empty($phones)) {
        usort($phones, function($a, $b) {
            return strcasecmp($a['name'], $b['name']);
        });
    }

    return $phones;
}

// 6. Fonction pour récupérer les contacts depuis les formulaires Contact Form 7
function wp_smsvertpro_get_cf7_contacts($with_names = false) {
    global $wpdb;
    $contacts = [];

    // Détecter quelle extension CF7 est utilisée et récupérer les données

    // 1. Flamingo (extension officielle CF7)
    $posts_table = $wpdb->prefix . 'posts';
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Checking if table exists
    if ($wpdb->get_var($wpdb->prepare('SHOW TABLES LIKE %s', $posts_table))) {
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- No WP equivalent for Flamingo plugin data
        $flamingo_contacts = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT p.ID, p.post_title
                FROM {$wpdb->prefix}posts p
                WHERE p.post_type = %s
                AND p.post_status = %s",
                'flamingo_inbound',
                'publish'
            ),
            ARRAY_A
        );

        foreach ($flamingo_contacts as $contact) {
            // Récupérer toutes les métadonnées
            $all_meta = get_post_meta($contact['ID']);

            // Flamingo stocke les champs dans des meta_keys individuels _field_XXX
            $phone = '';
            $name = $contact['post_title'];

            // Chercher le téléphone - chercher n'importe quel champ contenant tel/phone/mobile
            foreach ($all_meta as $meta_key => $meta_value) {
                // Vérifier si la clé commence par _field_ et contient tel/phone/mobile
                if (strpos($meta_key, '_field_') === 0 &&
                    (strpos($meta_key, 'tel') !== false ||
                     strpos($meta_key, 'phone') !== false ||
                     strpos($meta_key, 'mobile') !== false ||
                     strpos($meta_key, 'telephone') !== false)) {

                    if (is_array($meta_value) && !empty($meta_value[0])) {
                        $phone = $meta_value[0];
                        break;
                    }
                }
            }

            // Chercher prénom et nom
            $prenom = '';
            $nom = '';

            foreach ($all_meta as $meta_key => $meta_value) {
                if (strpos($meta_key, '_field_') === 0 && is_array($meta_value) && !empty($meta_value[0])) {
                    // Chercher prénom
                    if (strpos($meta_key, 'prenom') !== false ||
                        strpos($meta_key, 'first') !== false ||
                        strpos($meta_key, 'firstname') !== false) {
                        $prenom = $meta_value[0];
                    }

                    // Chercher nom
                    if (strpos($meta_key, 'nom') !== false && strpos($meta_key, 'prenom') === false ||
                        strpos($meta_key, 'last') !== false ||
                        strpos($meta_key, 'lastname') !== false) {
                        $nom = $meta_value[0];
                    }

                    // Chercher nom complet dans champs "name" ou "text-"
                    if (empty($prenom) && empty($nom)) {
                        if (strpos($meta_key, 'name') !== false ||
                            strpos($meta_key, 'text-') !== false) {
                            $name = $meta_value[0];
                        }
                    }
                }
            }

            // Combiner prénom et nom si trouvés
            if (!empty($prenom) || !empty($nom)) {
                $name = trim($prenom . ' ' . $nom);
            }

            // Utiliser _from_name si disponible et pas de nom trouvé
            if (empty($name) || $name === $contact['post_title']) {
                $from_name = get_post_meta($contact['ID'], '_from_name', true);
                if (!empty($from_name)) {
                    $name = $from_name;
                }
            }

            if (!empty($phone)) {
                $formatted = wp_smsvertpro_format_phone($phone);
                if (!empty($formatted)) {
                    if ($with_names) {
                        $contacts[] = ['phone' => $formatted, 'name' => $name];
                    } else {
                        $contacts[] = $formatted;
                    }
                }
            }
        }
    }

    // 2. Contact Form 7 Database Addon (CFDB7)
    $cfdb7_table = $wpdb->prefix . 'db7_forms';
    // Vérifier si la table existe de manière sécurisée
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Checking if table exists
    $table_exists = $wpdb->get_var($wpdb->prepare('SHOW TABLES LIKE %s', $cfdb7_table));
    if ($table_exists === $cfdb7_table) {
        // Utiliser une requête préparée avec le nom de table sécurisé
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- No WP equivalent for CFDB7 plugin data
        $cfdb7_data = $wpdb->get_results(
            "SELECT form_value, form_id FROM `{$wpdb->prefix}db7_forms`",
            ARRAY_A
        );

        foreach ($cfdb7_data as $row) {
            // Sécuriser la désérialisation en autorisant uniquement les tableaux
            // et en désactivant les classes PHP
            $data = @unserialize($row['form_value'], ['allowed_classes' => false]);

            // Validation stricte : vérifier que c'est bien un tableau
            if (!is_array($data)) {
                continue;
            }

            $phone = '';
            $name = '';

                // Chercher le téléphone - champs fixes + champs dynamiques (tel-XXX, phone-XXX)
                $phone_keys = ['phone', 'tel', 'telephone', 'mobile', 'tel-phone', 'your-phone', 'votre-telephone'];

                foreach (array_keys($data) as $field_key) {
                    if (strpos($field_key, 'tel-') === 0 || strpos($field_key, 'phone-') === 0) {
                        $phone_keys[] = $field_key;
                    }
                }

                foreach ($phone_keys as $key) {
                    if (isset($data[$key]) && !empty($data[$key])) {
                        $phone = $data[$key];
                        break;
                    }
                }

                // Chercher le nom - champs fixes + champs dynamiques
                $name_keys = ['your-name', 'name', 'nom', 'prenom'];

                foreach (array_keys($data) as $field_key) {
                    if (strpos($field_key, 'text-') === 0 || strpos($field_key, 'name-') === 0 || strpos($field_key, 'your-name') === 0) {
                        $name_keys[] = $field_key;
                    }
                }

                foreach ($name_keys as $key) {
                    if (isset($data[$key]) && !empty($data[$key])) {
                        $name = $data[$key];
                        break;
                    }
                }

                if (empty($name)) {
                    $name = 'Contact';
                }

            if (!empty($phone)) {
                $formatted = wp_smsvertpro_format_phone($phone);
                if (!empty($formatted)) {
                    if ($with_names) {
                        $contacts[] = ['phone' => $formatted, 'name' => $name];
                    } else {
                        $contacts[] = $formatted;
                    }
                }
            }
        }
    }

    // Retirer les doublons
    if ($with_names) {
        $unique_contacts = [];
        $seen_phones = [];
        foreach ($contacts as $contact) {
            if (!in_array($contact['phone'], $seen_phones)) {
                $unique_contacts[] = $contact;
                $seen_phones[] = $contact['phone'];
            }
        }

        // Trier par ordre alphabétique sur le nom
        usort($unique_contacts, function($a, $b) {
            return strcasecmp($a['name'], $b['name']);
        });

        return $unique_contacts;
    }

    return array_filter(array_unique($contacts));
}

// 7. Fonction pour parser une liste personnalisée de numéros
function wp_smsvertpro_parse_custom_numbers($text) {
    $phones = [];
    $lines = explode("\n", $text);

    foreach ($lines as $line) {
        $line = trim($line);
        if (!empty($line)) {
            $phones[] = wp_smsvertpro_format_phone($line);
        }
    }

    return array_filter(array_unique($phones));
}

// 7. Fonction pour formater un numéro de téléphone
function wp_smsvertpro_format_phone($phone) {
    // Nettoyer le numéro : retirer espaces, points, tirets, parenthèses
    $phone = preg_replace('/[^0-9+]/', '', $phone);

    // Si vide après nettoyage, retourner vide
    if (empty($phone)) {
        return '';
    }

    // Si commence par 06 ou 07 (format français local)
    if (preg_match('/^0[67][0-9]{8}$/', $phone)) {
        // Convertir en format international : retirer le 0 et ajouter 33
        return '33' . substr($phone, 1);
    }

    // Si commence déjà par 33 (format international)
    if (preg_match('/^33[67][0-9]{8}$/', $phone)) {
        return $phone;
    }

    // Si commence par +33
    if (preg_match('/^\+33[67][0-9]{8}$/', $phone)) {
        return substr($phone, 1); // Retirer le +
    }

    // Si commence par 00 33
    if (preg_match('/^0033[67][0-9]{8}$/', $phone)) {
        return substr($phone, 2); // Retirer 00
    }

    // Sinon retourner vide car format invalide
    return '';
}

// 8. Fonction pour envoyer des SMS en masse
function wp_smsvertpro_send_bulk_sms($phone_numbers, $message) {
    $results = [
        'total' => count($phone_numbers),
        'success' => 0,
        'failed' => 0,
        'errors' => []
    ];

    if (empty($phone_numbers)) {
        $results['errors'][] = 'Aucun numéro de téléphone trouvé.';
        return $results;
    }

    $email    = get_option('wp_smsvertpro_email');
    $password_encrypted = get_option('wp_smsvertpro_password');
    $sender   = get_option('wp_smsvertpro_sender');

    if (empty($email) || empty($password_encrypted)) {
        $results['errors'][] = 'Identifiants API manquants.';
        return $results;
    }

    // Déchiffrer le mot de passe
    $password = wp_smsvertpro_decrypt($password_encrypted);
    if (empty($password)) {
        $results['errors'][] = 'Erreur de déchiffrement du mot de passe.';
        return $results;
    }

    // Filtrer et valider les numéros
    $valid_phones = [];
    foreach ($phone_numbers as $phone) {
        if (preg_match('/^33[67][0-9]{8}$/', $phone)) {
            $valid_phones[] = $phone;
        } else {
            $results['failed']++;
            $results['errors'][] = "Numéro invalide : $phone";
        }
    }

    if (empty($valid_phones)) {
        $results['errors'][] = 'Aucun numéro valide à envoyer.';
        return $results;
    }

    // Créer un ID de campagne unique basé sur le timestamp
    $campaign_id = 'bulk_' . time();

    // Créer le flux XML avec TOUS les destinataires dans une seule requête
    $xml = '<?xml version="1.0" encoding="UTF-8"?>';
    $xml .= '<sms>';
    $xml .= '<login>';
    $xml .= '<user>' . esc_html($email) . '</user>';
    $xml .= '<pass>' . md5($password) . '</pass>';
    $xml .= '</login>';
    $xml .= '<message id="' . esc_attr($campaign_id) . '">';
    $xml .= '<sender>' . esc_html($sender) . '</sender>';
    $xml .= '<text>' . esc_html($message) . '</text>';
    $xml .= '<delay cancel=""></delay>';
    $xml .= '</message>';
    $xml .= '<recipients>';

    // Ajouter tous les destinataires
    foreach ($valid_phones as $phone) {
        $xml .= '<gsm id="">' . esc_html($phone) . '</gsm>';
    }

    $xml .= '</recipients>';
    $xml .= '</sms>';

    // Envoyer la requête à l'API
    $response = wp_remote_post(WP_SMSVERTPRO_API_URL . 'sending', [
        'timeout' => 30, // Augmenté pour les gros envois
        'headers' => [
            'Content-Type' => 'application/xml',
        ],
        'body' => $xml,
    ]);

    if (is_wp_error($response)) {
        $results['failed'] = count($valid_phones);
        $results['errors'][] = 'Erreur de connexion à l\'API : ' . $response->get_error_message();
        return $results;
    }

    $code = wp_remote_retrieve_response_code($response);
    if ($code !== 200) {
        $results['failed'] = count($valid_phones);
        $results['errors'][] = 'Erreur HTTP ' . $code;
        return $results;
    }

    // Parser la réponse XML
    $body = wp_remote_retrieve_body($response);
    libxml_use_internal_errors(true);
    $xml_response = simplexml_load_string($body);

    if ($xml_response === false) {
        $results['failed'] = count($valid_phones);
        $results['errors'][] = 'Erreur de parsing de la réponse XML';
        return $results;
    }

    // Vérifier le statut (SEND_OK)
    if (isset($xml_response->status) && (string)$xml_response->status === 'SEND_OK') {
        $results['success'] = count($valid_phones);
        return $results;
    } else {
        $results['failed'] = count($valid_phones);
        $error_status = isset($xml_response->status) ? (string)$xml_response->status : 'UNKNOWN';
        $results['errors'][] = 'Échec d\'envoi : ' . $error_status;
        return $results;
    }
}

// 9. Fonction pour récupérer les rapports/accusés de réception
function wp_smsvertpro_get_reports($campaign_id = '') {
    $email    = get_option('wp_smsvertpro_email');
    $password_encrypted = get_option('wp_smsvertpro_password');

    if (empty($email) || empty($password_encrypted)) {
        return false;
    }

    // Déchiffrer le mot de passe
    $password = wp_smsvertpro_decrypt($password_encrypted);
    if (empty($password)) {
        return false;
    }

    // Créer le flux XML
    $xml = '<?xml version="1.0" encoding="UTF-8"?>';
    $xml .= '<sms>';
    $xml .= '<login>';
    $xml .= '<user>' . esc_html($email) . '</user>';
    $xml .= '<pass>' . md5($password) . '</pass>';
    $xml .= '</login>';
    $xml .= '<report>';
    $xml .= '<message id="' . esc_attr($campaign_id) . '" sms="" />';
    $xml .= '</report>';
    $xml .= '</sms>';

    // Envoyer la requête POST
    $response = wp_remote_post(WP_SMSVERTPRO_API_URL . 'reports', [
        'timeout' => 15,
        'headers' => [
            'Content-Type' => 'application/xml',
        ],
        'body' => $xml,
    ]);

    if (is_wp_error($response)) {
        return false;
    }

    $code = wp_remote_retrieve_response_code($response);
    if ($code !== 200) {
        return false;
    }

    // Parser la réponse XML
    $body = wp_remote_retrieve_body($response);

    // Supprimer les caractères de contrôle invisibles
    $body = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/', '', $body);

    // Charger le XML
    libxml_use_internal_errors(true);
    $xml_response = simplexml_load_string($body);

    if ($xml_response === false) {
        return false;
    }

    // Parser les campagnes
    $campaigns = [];
    foreach ($xml_response->message as $message) {
        $campaign = [
            'id' => (string) $message['id'],
            'date' => (string) $message['date'],
            'nbsms' => (int) $message['nbsms'],
            'nbcredit' => (int) $message['nbcredit'],
            'sender' => (string) $message['sender'],
            'text' => (string) $message['text'],
            'recipients' => []
        ];

        // Parser tous les éléments <recipients> de ce <message>
        // Note: chaque <recipients> représente un destinataire
        if (isset($message->recipients)) {
            foreach ($message->recipients as $recipient) {
                $campaign['recipients'][] = [
                    'id' => (string) $recipient['id'],
                    'gsm' => (string) $recipient['gsm'],
                    'sentdate' => (string) $recipient['sentdate'],
                    'donedate' => (string) $recipient['donedate'],
                    'status' => (string) $recipient['status'],
                    'nbcreditsms' => (int) $recipient['nbcreditsms'],
                    'cancel' => (string) $recipient['cancel']
                ];
            }
        }

        $campaigns[] = $campaign;
    }

    return $campaigns;
}

// 10. Page : Historique des campagnes
function wp_smsvertpro_history_page() {
    if (!current_user_can('manage_options')) {
        return;
    }

    // Déconnexion
    if (isset($_POST['wp_smsvertpro_disconnect']) && check_admin_referer('wp_smsvertpro_disconnect')) {
        delete_option('wp_smsvertpro_email');
        delete_option('wp_smsvertpro_password');
        delete_option('wp_smsvertpro_sender');
        echo '<div class="notice notice-success is-dismissible"><p>Déconnecté avec succès.</p></div>';
    }

    $email    = get_option('wp_smsvertpro_email', '');
    $password = get_option('wp_smsvertpro_password', '');

    // Vérifier la connexion et récupérer les crédits
    $is_connected = false;
    $credits = null;
    if (!empty($email) && !empty($password)) {
        $credits = wp_smsvertpro_get_credits();
        if ($credits !== false) {
            $is_connected = true;
        }
    }

    // Récupérer les rapports
    $campaigns = [];
    if ($is_connected) {
        $campaigns = wp_smsvertpro_get_reports();
        if ($campaigns === false) {
            $campaigns = [];
        }
    }

    $logo_url = plugins_url('assets/smsvertpro-envoi-sms.png', __FILE__);
    ?>
    <style>
        .wrap .e-notice,
        .wrap .notice.is-dismissible:not(.notice-success):not(.notice-error) {
            display: none !important;
        }
        .campaign-card {
            background: #fff;
            border: 1px solid #ddd;
            border-radius: 4px;
            padding: 20px;
            margin-bottom: 20px;
        }
        .campaign-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
            padding-bottom: 15px;
            border-bottom: 1px solid #eee;
        }
        .campaign-title {
            font-size: 16px;
            font-weight: 600;
            color: #333;
        }
        .campaign-date {
            color: #666;
            font-size: 14px;
        }
        .campaign-stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 15px;
            margin-bottom: 15px;
        }
        .stat-box {
            background: #f9f9f9;
            padding: 10px;
            border-radius: 4px;
            text-align: center;
        }
        .stat-label {
            font-size: 12px;
            color: #666;
            text-transform: uppercase;
        }
        .stat-value {
            font-size: 20px;
            font-weight: 600;
            color: #2271b1;
        }
        .campaign-message {
            background: #f0f0f1;
            padding: 15px;
            border-left: 4px solid #2271b1;
            margin-bottom: 15px;
            font-style: italic;
        }
        .recipients-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 15px;
        }
        .recipients-table th,
        .recipients-table td {
            padding: 10px;
            text-align: left;
            border-bottom: 1px solid #eee;
        }
        .recipients-table th {
            background: #f9f9f9;
            font-weight: 600;
            font-size: 12px;
            text-transform: uppercase;
            color: #666;
        }
        .status-badge {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 600;
        }
        .status-delivered {
            background: #d4edda;
            color: #155724;
        }
        .status-pending {
            background: #fff3cd;
            color: #856404;
        }
        .status-failed {
            background: #f8d7da;
            color: #721c24;
        }
        .status-invalid {
            background: #e2e3e5;
            color: #383d41;
        }
        .no-campaigns {
            text-align: center;
            padding: 60px 20px;
            background: #fff;
            border: 1px solid #ddd;
            border-radius: 4px;
        }
        .no-campaigns-icon {
            font-size: 48px;
            color: #ccc;
            margin-bottom: 20px;
        }
    </style>
    <div class="wrap">
        <?php if (!$is_connected): ?>
            <div style="text-align: center; margin: 40px 0;">
                <img src="<?php echo esc_url($logo_url); ?>" alt="SMS Vert Pro" style="max-height: 80px;">
            </div>
            <div style="max-width: 600px; margin: 0 auto;">
                <div class="card">
                    <h2>Connexion requise</h2>
                    <p>Vous devez d'abord vous connecter à votre compte SMS Vert Pro pour accéder à l'historique des campagnes.</p>
                    <p><a href="<?php echo esc_url(admin_url('admin.php?page=smsvertpro')); ?>" class="button button-primary">Se connecter</a></p>
                </div>
            </div>
        <?php else: ?>
            <div style="text-align: center; margin: 20px 0;">
                <img src="<?php echo esc_url($logo_url); ?>" alt="SMS Vert Pro" style="max-height: 60px;">
            </div>

            <div style="max-width: 1200px; margin: 0 auto; padding: 0 20px;">
                <div class="notice notice-success" style="display: flex; align-items: center; justify-content: space-between;">
                    <p>
                        <strong>✓ Connecté</strong> &nbsp;|&nbsp;
                        <strong>Crédits SMS disponibles : <?php echo esc_html($credits); ?></strong> &nbsp;|&nbsp;
                        Compte : <?php echo esc_html($email); ?>
                    </p>
                    <form method="post" style="margin: 0;">
                        <?php wp_nonce_field('wp_smsvertpro_disconnect'); ?>
                        <button type="submit" name="wp_smsvertpro_disconnect" class="button">Se déconnecter</button>
                    </form>
                </div>

                <h2>Historique des campagnes SMS</h2>

                <?php if (empty($campaigns)): ?>
                    <div class="no-campaigns">
                        <div class="no-campaigns-icon">📭</div>
                        <h3>Aucune campagne trouvée</h3>
                        <p>Vous n'avez pas encore envoyé de campagne SMS ou aucun rapport n'est disponible.</p>
                        <p><a href="<?php echo esc_url(admin_url('admin.php?page=smsvertpro')); ?>" class="button button-primary">Envoyer un SMS</a></p>
                    </div>
                <?php else: ?>
                    <?php foreach ($campaigns as $campaign): ?>
                        <div class="campaign-card">
                            <div class="campaign-header">
                                <div>
                                    <div class="campaign-title">
                                        <?php echo esc_html($campaign['sender']); ?>
                                        <?php if (!empty($campaign['id'])): ?>
                                            - Campagne #<?php echo esc_html($campaign['id']); ?>
                                        <?php endif; ?>
                                    </div>
                                    <div class="campaign-date">
                                        <?php
                                        if (!empty($campaign['date'])) {
                                            $date_parts = explode('/', $campaign['date']);
                                            if (count($date_parts) === 2) {
                                                list($date, $time) = $date_parts;
                                                echo esc_html($date . ' à ' . $time);
                                            } else {
                                                echo esc_html($campaign['date']);
                                            }
                                        }
                                        ?>
                                    </div>
                                </div>
                            </div>

                            <div class="campaign-stats">
                                <div class="stat-box">
                                    <div class="stat-label">SMS envoyés</div>
                                    <div class="stat-value"><?php echo absint($campaign['nbsms']); ?></div>
                                </div>
                                <div class="stat-box">
                                    <div class="stat-label">Crédits utilisés</div>
                                    <div class="stat-value"><?php echo absint($campaign['nbcredit']); ?></div>
                                </div>
                                <div class="stat-box">
                                    <div class="stat-label">Délivrés</div>
                                    <div class="stat-value" style="color: #28a745;">
                                        <?php
                                        $delivered = 0;
                                        foreach ($campaign['recipients'] as $recipient) {
                                            if ($recipient['status'] === 'DELIVERED') {
                                                $delivered++;
                                            }
                                        }
                                        echo absint($delivered);
                                        ?>
                                    </div>
                                </div>
                                <div class="stat-box">
                                    <div class="stat-label">En attente / Échec</div>
                                    <div class="stat-value" style="color: #dc3545;">
                                        <?php echo absint($campaign['nbsms'] - $delivered); ?>
                                    </div>
                                </div>
                            </div>

                            <?php if (!empty($campaign['text'])): ?>
                                <div class="campaign-message">
                                    <strong>Message :</strong> <?php echo esc_html($campaign['text']); ?>
                                </div>
                            <?php endif; ?>

                            <?php if (!empty($campaign['recipients'])): ?>
                                <details>
                                    <summary style="cursor: pointer; font-weight: 600; padding: 10px 0;">
                                        📋 Voir le détail des <?php echo count($campaign['recipients']); ?> destinataire(s)
                                    </summary>
                                    <table class="recipients-table">
                                        <thead>
                                            <tr>
                                                <th>Numéro</th>
                                                <th>Envoyé le</th>
                                                <th>Délivré le</th>
                                                <th>Statut</th>
                                                <th>Crédits</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($campaign['recipients'] as $recipient): ?>
                                                <tr>
                                                    <td>
                                                        <?php
                                                        $gsm = $recipient['gsm'];
                                                        if (strlen($gsm) === 11) {
                                                            $formatted_phone = '+' . substr($gsm, 0, 2) . ' ' . substr($gsm, 2, 1) . ' ' .
                                                                 substr($gsm, 3, 2) . ' ' . substr($gsm, 5, 2) . ' ' .
                                                                 substr($gsm, 7, 2) . ' ' . substr($gsm, 9, 2);
                                                            echo esc_html($formatted_phone);
                                                        } else {
                                                            echo esc_html($gsm);
                                                        }
                                                        ?>
                                                    </td>
                                                    <td><?php echo esc_html($recipient['sentdate']); ?></td>
                                                    <td>
                                                        <?php
                                                        if ($recipient['status'] === 'DELIVERED' && !empty($recipient['donedate'])) {
                                                            echo esc_html($recipient['donedate']);
                                                        } else {
                                                            echo '-';
                                                        }
                                                        ?>
                                                    </td>
                                                    <td>
                                                        <?php
                                                        $status = $recipient['status'];
                                                        $status_class = 'status-pending';
                                                        $status_label = 'En attente';

                                                        if ($status === 'DELIVERED') {
                                                            $status_class = 'status-delivered';
                                                            $status_label = '✓ Délivré';
                                                        } elseif ($status === 'NOT_DELIVERED') {
                                                            $status_class = 'status-failed';
                                                            $status_label = '✗ Non délivré';
                                                        } elseif ($status === 'INVALID_DESTINATION_ADDRESS') {
                                                            $status_class = 'status-invalid';
                                                            $status_label = '⚠ Numéro invalide';
                                                        } elseif ($status === 'NULL' || empty($status)) {
                                                            $status_class = 'status-pending';
                                                            $status_label = '⏳ En attente';
                                                        }
                                                        ?>
                                                        <span class="status-badge <?php echo esc_attr($status_class); ?>">
                                                            <?php echo esc_html($status_label); ?>
                                                        </span>
                                                    </td>
                                                    <td><?php echo absint($recipient['nbcreditsms']); ?></td>
                                                </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </details>
                            <?php endif; ?>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    </div>
    <?php
}

// ======================================================================
// SYSTÈME DE MISE À JOUR AUTOMATIQUE
// ======================================================================

/**
* Vérifie les mises à jour du plugin depuis le serveur distant
*/

function wp_smsvertpro_check_for_update($transient) {
    if (empty($transient->checked)) {
        return $transient;
    }

    // Définir le slug du plugin
    $plugin_slug = plugin_basename(__FILE__);

    // Récupérer les informations de mise à jour depuis le serveur
    $remote_info = wp_smsvertpro_get_remote_plugin_info();

    if ($remote_info === false) {
        return $transient;
    }

    // Comparer les versions
    if (version_compare(WP_SMSVERTPRO_VERSION, $remote_info->version, '<')) {
        // Une mise à jour est disponible
        $transient->response[$plugin_slug] = (object) [
            'slug'        => dirname($plugin_slug),
            'plugin'      => $plugin_slug,
            'new_version' => $remote_info->version,
            'url'         => $remote_info->homepage,
            'package'     => $remote_info->download_url,
            'tested'      => $remote_info->tested,
            'requires'    => $remote_info->requires,
            'requires_php' => $remote_info->requires_php,
        ];
    }

    return $transient;
}
add_filter('pre_set_site_transient_update_plugins', 'wp_smsvertpro_check_for_update');

/**
    * Affiche les informations du plugin dans la popup de mise à jour
    */
function wp_smsvertpro_plugin_info($false, $action, $args) {
    $plugin_slug = dirname(plugin_basename(__FILE__));

    // Vérifier si c'est bien notre plugin
    if ($action !== 'plugin_information' || $args->slug !== $plugin_slug) {
        return $false;
    }

    // Récupérer les informations depuis le serveur
    $remote_info = wp_smsvertpro_get_remote_plugin_info();

    if ($remote_info === false) {
        return $false;
    }

    // Construire l'objet de réponse
    $plugin_info = (object) [
        'name'          => $remote_info->name,
        'slug'          => $plugin_slug,
        'version'       => $remote_info->version,
        'author'        => $remote_info->author,
        'homepage'      => $remote_info->homepage,
        'requires'      => $remote_info->requires,
        'tested'        => $remote_info->tested,
        'requires_php'  => $remote_info->requires_php,
        'download_link' => $remote_info->download_url,
        'sections'      => [
            'description'  => $remote_info->sections->description,
            'installation' => $remote_info->sections->installation,
            'changelog'    => $remote_info->sections->changelog,
        ],
        'banners' => [
            'low'  => isset($remote_info->banners->low) ? $remote_info->banners->low : '',
            'high' => isset($remote_info->banners->high) ? $remote_info->banners->high : '',
        ],
    ];

    return $plugin_info;
}
add_filter('plugins_api', 'wp_smsvertpro_plugin_info', 20, 3);

/**
    * Récupère les informations du plugin depuis le serveur distant
    *
    * @return object|false Les informations du plugin ou false en cas d'erreur
    */
function wp_smsvertpro_get_remote_plugin_info() {
    // Vérifier le cache (durée: 12 heures)
    $cache_key = 'wp_smsvertpro_update_info';
    $cached_info = get_transient($cache_key);

    if ($cached_info !== false) {
        return $cached_info;
    }

    // Récupérer les informations depuis le serveur
    $response = wp_remote_get(WP_SMSVERTPRO_UPDATE_URL, [
        'timeout' => 10,
        'headers' => [
            'Accept' => 'application/json',
        ],
    ]);

    if (is_wp_error($response)) {
        return false;
    }

    $code = wp_remote_retrieve_response_code($response);
    if ($code !== 200) {
        return false;
    }

    $body = wp_remote_retrieve_body($response);
    $remote_info = json_decode($body);

    if (json_last_error() !== JSON_ERROR_NONE || !is_object($remote_info)) {
        return false;
    }

    // Valider la structure des données
    if (!isset($remote_info->version) || !isset($remote_info->download_url)) {
        return false;
    }

    // Mettre en cache pour 12 heures
    set_transient($cache_key, $remote_info, 12 * HOUR_IN_SECONDS);

    return $remote_info;
}

/**
    * Supprime le cache des informations de mise à jour après une mise à jour
    */
function wp_smsvertpro_purge_update_cache($upgrader, $options) {
    if ($options['action'] === 'update' && $options['type'] === 'plugin') {
        delete_transient('wp_smsvertpro_update_info');
    }
}
add_action('upgrader_process_complete', 'wp_smsvertpro_purge_update_cache', 10, 2);